<?php
define('APP_ACCESS', true);

if (!file_exists('config.php')) {
    header('Content-Type: application/json');
    http_response_code(500);
    echo json_encode(['error' => 'Configuration file missing']);
    exit;
}
require_once 'config.php';

// Set CORS and security headers
Utils::setCORSHeaders();
Utils::setSecurityHeaders();

// Get request information
$requestMethod = $_SERVER['REQUEST_METHOD'];
$requestUri = $_SERVER['REQUEST_URI'];
$pathInfo = parse_url($requestUri, PHP_URL_PATH);

// Remove backend path prefix if present
$pathInfo = preg_replace('#^/backend#', '', $pathInfo);

// Parse the path
$pathSegments = array_filter(explode('/', trim($pathInfo, '/')));

$endpoint = $pathSegments[0] ?? '';

// Different logic for different endpoints
if ($endpoint === 'admin') {
    // For admin routes: /admin/login, /admin/verify-token, etc.
    $action = $pathSegments[1] ?? '';  // "login", "verify-token"
    $id = $pathSegments[2] ?? '';      // additional parameter if needed
} elseif ($endpoint === 'products') {
    // For product routes: /products/123, /products (list)
    $id = $pathSegments[1] ?? '';      // product ID like "123"
    $action = $pathSegments[2] ?? '';  // additional action if needed
} elseif ($endpoint === 'analytics') {
    // For analytics routes: /analytics/realtime, /analytics/export
    $action = $pathSegments[1] ?? '';  // "realtime", "export", etc.
    $id = $pathSegments[2] ?? '';      // additional parameter if needed
} else {
    // Default logic for other endpoints
    $id = $pathSegments[1] ?? '';
    $action = $pathSegments[2] ?? '';
}

// Clean the values
$endpoint = preg_replace('/[^a-zA-Z0-9_-]/', '', $endpoint);
$action = preg_replace('/[^a-zA-Z0-9_-]/', '', $action);
$id = preg_replace('/[^a-zA-Z0-9_-]/', '', $id);

// For products endpoint, ensure ID is numeric if it's meant to be an ID
if ($endpoint === 'products' && !empty($id) && !is_numeric($id)) {
    // If it's not numeric, it might be an action, so swap them
    $temp = $action;
    $action = $id;
    $id = $temp;
}

// Get request body for POST/PUT requests
$requestBody = [];
if (in_array($requestMethod, ['POST', 'PUT', 'PATCH'])) {
    $input = file_get_contents('php://input');
    $requestBody = json_decode($input, true) ?? [];
    if (empty($requestBody) && empty($_POST)) {
        Utils::respondJSON(['error' => 'Invalid or empty request body'], 400);
    }
    // Merge with $_POST for form data
    $requestBody = array_merge($_POST, $requestBody);
}

// Merge GET parameters
$requestData = array_merge($_GET, $requestBody);

// Log request for debugging
Utils::logError("Request: $requestMethod $pathInfo - Parsed: endpoint=$endpoint, action=$action, id=$id", $requestData);

// Basic routing
try {
    switch ($endpoint) {
        case 'products':
            if (!file_exists('api/products.php')) {
                Utils::respondJSON(['error' => 'Products handler not found'], 500);
            }
            require_once 'api/products.php';
            handleProductsRequest($requestMethod, $action, $id, $requestData);
            break;

        case 'categories':
            if (!file_exists('api/categories.php')) {
                Utils::respondJSON(['error' => 'Categories handler not found'], 500);
            }
            require_once 'api/categories.php';
            handleCategoriesRequest($requestMethod, $action, $id, $requestData);
            break;

        case 'admin':
            if (!file_exists('api/admin.php')) {
                Utils::respondJSON(['error' => 'Admin handler not found'], 500);
            }
            require_once 'api/admin.php';
            // Pass action correctly for admin routes
            handleAdminRequest($requestMethod, $action, $id, $requestData);
            break;

        case 'analytics':
            if (!file_exists('api/analytics.php')) {
                Utils::respondJSON(['error' => 'Analytics handler not found'], 500);
            }
            require_once 'api/analytics.php';
            handleAnalyticsRequest($requestMethod, $action, $id, $requestData);
            break;

        case 'upload':
            if (!file_exists('api/upload.php')) {
                Utils::respondJSON(['error' => 'Upload handler not found'], 500);
            }
            require_once 'api/upload.php';
            handleUploadRequest($requestMethod, $action, $id, $requestData);
            break;

        case 'settings':
            if (!file_exists('api/settings.php')) {
                Utils::respondJSON(['error' => 'Settings handler not found'], 500);
            }
            require_once 'api/settings.php';
            handleSettingsRequest($requestMethod, $action, $id, $requestData);
            break;

        case 'health':
            Utils::respondJSON([
                'status' => 'ok',
                'timestamp' => time(),
                'version' => APP_VERSION,
                'database' => 'connected' // Replace with actual DB check
            ]);
            break;

        default:
            Utils::respondJSON([
                'error' => 'Endpoint not found',
                'available_endpoints' => [
                    'products',
                    'categories',
                    'admin',
                    'analytics',
                    'upload',
                    'settings',
                    'health'
                ]
            ], 404);
    }
} catch (Exception $e) {
    Utils::logError('API Error: ' . $e->getMessage(), [
        'endpoint' => $endpoint,
        'method' => $requestMethod,
        'data' => $requestData
    ]);
    Utils::respondJSON([
        'error' => APP_DEBUG ? $e->getMessage() : 'Internal server error'
    ], 500);
}
?>