<?php
// Prevent direct access
if (!defined('APP_ACCESS')) {
    die('Direct access not permitted');
}

// Error reporting for development (set to 0 for production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Application Configuration
define('APP_NAME', 'PreOrder Store');
define('APP_VERSION', '1.0.0');
define('APP_DEBUG', true); // Set to false in production
define('APP_TIMEZONE', 'UTC');

// Database Configuration
define('DB_HOST', 'localhost');
define('DB_NAME', 'bhacicyg_preorder');
define('DB_USER', 'bhacicyg_preorder_user'); // Change for production
define('DB_PASS', 'preorder_user'); // Set password for production
define('DB_CHARSET', 'utf8mb4');

// JWT Configuration - MAKE SURE THIS IS EXACTLY 32+ CHARACTERS
define('JWT_SECRET', 'your-super-secret-jwt-key-change-in-production-min-32-chars-12345');
define('JWT_EXPIRY', 86400); // 24 hours in seconds

// File Upload Configuration
define('UPLOAD_DIR', __DIR__ . '/uploads/');
define('UPLOAD_URL', '/uploads/');
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_EXTENSIONS', ['jpg', 'jpeg', 'png', 'webp', 'gif']);

// API Configuration
define('API_RATE_LIMIT', 100); // Requests per minute
define('API_TIMEOUT', 30); // Seconds

// WhatsApp Configuration
define('WHATSAPP_NUMBER', '+1234567890'); // Replace with actual business number
define('WHATSAPP_MESSAGE_TEMPLATE', "Hello! I'd like to place an order:\n\n");

// CORS Configuration
define('ALLOWED_ORIGINS', [
    'http://127.0.0.1:5500',
    'http://localhost:5500',
    'http://preorder.api', 
    'http://192.168.1.37:5500', 
    'http://localhost:3000',
    'capacitor://localhost', 
    'https://your-domain.com' ,
    'http://192.168.1.36:8000',
    'http://10.0.2.2:8000'  
]);

// Security Headers
define('SECURITY_HEADERS', [
    'X-Content-Type-Options: nosniff',
    'X-Frame-Options: DENY',
    'X-XSS-Protection: 1; mode=block',
    'Referrer-Policy: strict-origin-when-cross-origin'
]);

// Cache Configuration
define('CACHE_ENABLED', true);
define('CACHE_DURATION', 3600); // 1 hour

// Pagination Configuration
define('DEFAULT_PAGE_SIZE', 12);
define('MAX_PAGE_SIZE', 50);

// Email Configuration (for future use)
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_USERNAME', '');
define('SMTP_PASSWORD', '');
define('FROM_EMAIL', 'noreply@preorderstore.com');
define('FROM_NAME', 'PreOrder Store');

// Image Processing Configuration
define('IMAGE_QUALITY', 85);
define('THUMBNAIL_WIDTH', 300);
define('THUMBNAIL_HEIGHT', 300);
define('LARGE_IMAGE_WIDTH', 800);
define('LARGE_IMAGE_HEIGHT', 600);

// Database Connection Class
class Database {
    private static $instance = null;
    private $connection;
    
    private function __construct() {
        try {
            $this->connection = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET,
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false,
                    PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET,
                ]
            );
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            die(json_encode(['error' => 'Database connection failed']));
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->connection;
    }
}

// Utility Functions
class Utils {
    public static function sanitizeInput($input) {
        if (is_array($input)) {
            return array_map([self::class, 'sanitizeInput'], $input);
        }
        return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
    }
    
    public static function validateEmail($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL);
    }
    
    public static function generateUUID() {
        return bin2hex(random_bytes(16));
    }
    
    public static function formatPrice($price) {
        return number_format((float)$price, 2);
    }
    
    public static function createSlug($string) {
        $slug = preg_replace('/[^a-zA-Z0-9\s-]/', '', $string);
        $slug = preg_replace('/\s+/', '-', trim($slug));
        return strtolower($slug);
    }
    
    public static function resizeImage($source, $destination, $maxWidth, $maxHeight, $quality = 85) {
        $imageInfo = getimagesize($source);
        if (!$imageInfo) return false;
        
        $width = $imageInfo[0];
        $height = $imageInfo[1];
        $type = $imageInfo[2];
        
        // Calculate new dimensions
        $ratio = min($maxWidth / $width, $maxHeight / $height);
        $newWidth = (int)($width * $ratio);
        $newHeight = (int)($height * $ratio);
        
        // Create image resource
        switch ($type) {
            case IMAGETYPE_JPEG:
                $sourceImage = imagecreatefromjpeg($source);
                break;
            case IMAGETYPE_PNG:
                $sourceImage = imagecreatefrompng($source);
                break;
            case IMAGETYPE_WEBP:
                $sourceImage = imagecreatefromwebp($source);
                break;
            default:
                return false;
        }
        
        if (!$sourceImage) return false;
        
        // Create new image
        $newImage = imagecreatetruecolor($newWidth, $newHeight);
        
        // Preserve transparency for PNG
        if ($type == IMAGETYPE_PNG) {
            imagealphablending($newImage, false);
            imagesavealpha($newImage, true);
        }
        
        // Resize
        imagecopyresampled(
            $newImage, $sourceImage,
            0, 0, 0, 0,
            $newWidth, $newHeight,
            $width, $height
        );
        
        // Save
        $result = false;
        switch ($type) {
            case IMAGETYPE_JPEG:
                $result = imagejpeg($newImage, $destination, $quality);
                break;
            case IMAGETYPE_PNG:
                $result = imagepng($newImage, $destination);
                break;
            case IMAGETYPE_WEBP:
                $result = imagewebp($newImage, $destination, $quality);
                break;
        }
        
        imagedestroy($sourceImage);
        imagedestroy($newImage);
        
        return $result;
    }
    
    public static function logError($message, $context = []) {
        $log = date('Y-m-d H:i:s') . " - " . $message;
        if (!empty($context)) {
            $log .= " - Context: " . json_encode($context);
        }
        error_log($log . "\n", 3, __DIR__ . '/logs/error.log');
    }
    
    public static function respondJSON($data, $statusCode = 200) {
        http_response_code($statusCode);
        header('Content-Type: application/json');
        echo json_encode($data);
        exit;
    }
    
    public static function validateRequiredFields($data, $requiredFields) {
        $missing = [];
        foreach ($requiredFields as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                $missing[] = $field;
            }
        }
        return $missing;
    }
    
    public static function setCORSHeaders() {
        $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
        
        if (APP_DEBUG) {
            // In debug mode, allow all origins for easier testing
            header('Access-Control-Allow-Origin: *');
        } elseif ($origin && in_array($origin, ALLOWED_ORIGINS)) {
            header("Access-Control-Allow-Origin: $origin");
        }
        
        header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
        header('Access-Control-Allow-Credentials: true');
        header('Access-Control-Max-Age: 86400');
        
        // Handle preflight requests
        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            http_response_code(204);
            exit;
        }
    }
    
    public static function setSecurityHeaders() {
        foreach (SECURITY_HEADERS as $header) {
            header($header);
        }
    }

    public static function getClientIP() {
        $ipKeys = [
            'HTTP_CF_CONNECTING_IP', 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED', 'REMOTE_ADDR'
        ];
        
        foreach ($ipKeys as $key) {
            if (!empty($_SERVER[$key])) {
                $ips = explode(',', $_SERVER[$key]);
                foreach ($ips as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP)) {
                        return $ip;
                    }
                }
            }
        }
        
        return '127.0.0.1';
    }
}

// JWT Helper Class with DEBUG
class JWT {
    public static function encode($payload) {
        Utils::logError('JWT encode called', [
            'payload' => $payload,
            'secret_length' => strlen(JWT_SECRET)
        ]);
        
        $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
        $payload = json_encode($payload);
        
        $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
        $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
        
        $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, JWT_SECRET, true);
        $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        $token = $base64Header . "." . $base64Payload . "." . $base64Signature;
        
        Utils::logError('JWT token created', [
            'token_length' => strlen($token),
            'header_length' => strlen($base64Header),
            'payload_length' => strlen($base64Payload),
            'signature_length' => strlen($base64Signature)
        ]);
        
        return $token;
    }
    
    // public static function decode($jwt) {
    //     Utils::logError('JWT decode called', [
    //         'jwt_length' => strlen($jwt),
    //         'jwt_parts' => explode('.', $jwt)
    //     ]);
        
    //     $parts = explode('.', $jwt);
        
    //     if (count($parts) !== 3) {
    //         Utils::logError('JWT decode failed: invalid parts count', [
    //             'parts_count' => count($parts),
    //             'expected' => 3
    //         ]);
    //         return false;
    //     }
        
    //     $header = base64_decode(str_replace(['-', '_'], ['+', '/'], $parts[0]));
    //     $payload = base64_decode(str_replace(['-', '_'], ['+', '/'], $parts[1]));
    //     $signature = str_replace(['-', '_'], ['+', '/'], $parts[2]);
        
    //     Utils::logError('JWT decode parts extracted', [
    //         'header_decoded' => $header,
    //         'payload_decoded' => $payload,
    //         'signature_length' => strlen($signature)
    //     ]);
        
    //     $expectedSignature = hash_hmac('sha256', $parts[0] . "." . $parts[1], JWT_SECRET, true);
    //     $base64ExpectedSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($expectedSignature));
        
    //     Utils::logError('JWT signature verification', [
    //         'received_signature' => $signature,
    //         'expected_signature' => $base64ExpectedSignature,
    //         'signatures_match' => hash_equals($signature, $base64ExpectedSignature),
    //         'secret_used' => substr(JWT_SECRET, 0, 10) . '...'
    //     ]);
        
    //     if (!hash_equals($signature, $base64ExpectedSignature)) {
    //         Utils::logError('JWT signature verification failed');
    //         return false;
    //     }
        
    //     $data = json_decode($payload, true);
        
    //     if (!$data) {
    //         Utils::logError('JWT payload decode failed', [
    //             'payload_raw' => $payload,
    //             'json_error' => json_last_error_msg()
    //         ]);
    //         return false;
    //     }
        
    //     Utils::logError('JWT payload decoded successfully', [
    //         'user_id' => $data['user_id'] ?? 'not set',
    //         'exp' => $data['exp'] ?? 'not set',
    //         'current_time' => time(),
    //         'is_expired' => isset($data['exp']) ? ($data['exp'] < time()) : 'no exp set'
    //     ]);
        
    //     if (isset($data['exp']) && $data['exp'] < time()) {
    //         Utils::logError('JWT token expired', [
    //             'exp' => $data['exp'],
    //             'current_time' => time(),
    //             'expired_by' => time() - $data['exp'] . ' seconds'
    //         ]);
    //         return false;
    //     }
        
    //     return $data;
    // }
    
    public static function decode($jwt) {
    Utils::logError('JWT decode called', [
        'jwt_length' => strlen($jwt),
        'jwt_parts' => explode('.', $jwt)
    ]);
    
    $parts = explode('.', $jwt);
    
    if (count($parts) !== 3) {
        Utils::logError('JWT decode failed: invalid parts count', [
            'parts_count' => count($parts),
            'expected' => 3
        ]);
        return false;
    }
    
    // Decode header and payload
    $header = base64_decode(str_replace(['-', '_'], ['+', '/'], $parts[0]));
    $payload = base64_decode(str_replace(['-', '_'], ['+', '/'], $parts[1]));
    
    // Get the received signature (keep as URL-safe base64)
    $receivedSignature = $parts[2];
    
    Utils::logError('JWT decode parts extracted', [
        'header_decoded' => $header,
        'payload_decoded' => $payload,
        'signature_length' => strlen($receivedSignature)
    ]);
    
    // Create expected signature using the same format as encoding
    $expectedSignature = hash_hmac('sha256', $parts[0] . "." . $parts[1], JWT_SECRET, true);
    $base64ExpectedSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($expectedSignature));
    
    Utils::logError('JWT signature verification', [
        'received_signature' => $receivedSignature,
        'expected_signature' => $base64ExpectedSignature,
        'signatures_match' => hash_equals($receivedSignature, $base64ExpectedSignature),
        'secret_used' => substr(JWT_SECRET, 0, 10) . '...'
    ]);
    
    // Compare signatures directly (both should be URL-safe base64)
    if (!hash_equals($receivedSignature, $base64ExpectedSignature)) {
        Utils::logError('JWT signature verification failed');
        return false;
    }
    
    $data = json_decode($payload, true);
    
    if (!$data) {
        Utils::logError('JWT payload decode failed', [
            'payload_raw' => $payload,
            'json_error' => json_last_error_msg()
        ]);
        return false;
    }
    
    Utils::logError('JWT payload decoded successfully', [
        'user_id' => $data['user_id'] ?? 'not set',
        'exp' => $data['exp'] ?? 'not set',
        'current_time' => time(),
        'is_expired' => isset($data['exp']) ? ($data['exp'] < time()) : 'no exp set'
    ]);
    
    if (isset($data['exp']) && $data['exp'] < time()) {
        Utils::logError('JWT token expired', [
            'exp' => $data['exp'],
            'current_time' => time(),
            'expired_by' => time() - $data['exp'] . ' seconds'
        ]);
        return false;
    }
    
    return $data;
}

    public static function createToken($userId, $username) {
        $payload = [
            'user_id' => $userId,
            'username' => $username,
            'iat' => time(),
            'exp' => time() + JWT_EXPIRY
        ];
        
        Utils::logError('Creating JWT token', $payload);
        
        return self::encode($payload);
    }
    
    public static function validateToken($token) {
        Utils::logError('Validating JWT token', [
            'token_provided' => !empty($token),
            'token_length' => strlen($token ?? ''),
            'secret_length' => strlen(JWT_SECRET)
        ]);
        
        if (!$token) {
            Utils::logError('JWT validation failed: no token provided');
            return false;
        }
        
        // Remove "Bearer " prefix if present
        if (strpos($token, 'Bearer ') === 0) {
            $token = substr($token, 7);
            Utils::logError('Removed Bearer prefix from token');
        }
        
        return self::decode($token);
    }
}

// Initialize timezone
date_default_timezone_set(APP_TIMEZONE);

// Create necessary directories
$directories = [
    __DIR__ . '/uploads',
    __DIR__ . '/logs',
    __DIR__ . '/cache'
];

foreach ($directories as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// Set up error logging
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/logs/php_errors.log');

// Validate upload directory
if (!is_dir(UPLOAD_DIR) || !is_writable(UPLOAD_DIR)) {
    error_log('Upload directory is not writable: ' . UPLOAD_DIR);
}

// Log JWT configuration on startup
Utils::logError('JWT Configuration Check', [
    'JWT_SECRET_length' => strlen(JWT_SECRET),
    'JWT_EXPIRY' => JWT_EXPIRY,
    'current_time' => time()
]);
?>