<?php

if (!defined('APP_ACCESS')) {
    die('Direct access not permitted');
}

function handleUploadRequest($method, $action, $id, $data) {
    // Validate admin token
    $token = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    $user = JWT::validateToken($token);
    
    if (!$user) {
        Utils::respondJSON(['error' => 'Unauthorized'], 401);
    }
    
    switch ($method) {
        case 'POST':
            if ($action === 'image' || $action === '') {
                uploadImage();
            } elseif ($action === 'multiple') {
                uploadMultipleImages();
            } else {
                Utils::respondJSON(['error' => 'Invalid upload action'], 400);
            }
            break;
            
        case 'DELETE':
            if ($id) {
                deleteImage($id);
            } else {
                Utils::respondJSON(['error' => 'Image path required'], 400);
            }
            break;
            
        default:
            Utils::respondJSON(['error' => 'Method not allowed'], 405);
    }
}

function uploadImage() {
    try {
        // Check if file was uploaded
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            $errorMessages = [
                UPLOAD_ERR_INI_SIZE => 'File too large (server limit)',
                UPLOAD_ERR_FORM_SIZE => 'File too large (form limit)', 
                UPLOAD_ERR_PARTIAL => 'File partially uploaded',
                UPLOAD_ERR_NO_FILE => 'No file uploaded',
                UPLOAD_ERR_NO_TMP_DIR => 'No temporary directory',
                UPLOAD_ERR_CANT_WRITE => 'Failed to write file',
                UPLOAD_ERR_EXTENSION => 'Upload stopped by extension'
            ];
            
            $error = $_FILES['file']['error'] ?? UPLOAD_ERR_NO_FILE;
            $message = $errorMessages[$error] ?? 'Unknown upload error';
            
            Utils::respondJSON(['error' => $message], 400);
        }
        
        $file = $_FILES['file'];
        
        // Validate file size
        if ($file['size'] > MAX_FILE_SIZE) {
            $maxSizeMB = MAX_FILE_SIZE / (1024 * 1024);
            Utils::respondJSON(['error' => "File too large. Maximum size: {$maxSizeMB}MB"], 400);
        }
        
        // Validate file type
        $fileInfo = pathinfo($file['name']);
        $extension = strtolower($fileInfo['extension'] ?? '');
        
        if (!in_array($extension, ALLOWED_EXTENSIONS)) {
            $allowedStr = implode(', ', ALLOWED_EXTENSIONS);
            Utils::respondJSON(['error' => "Invalid file type. Allowed: {$allowedStr}"], 400);
        }
        
        // Validate image using getimagesize
        $imageInfo = getimagesize($file['tmp_name']);
        if (!$imageInfo) {
            Utils::respondJSON(['error' => 'Invalid image file'], 400);
        }
        
        $mimeType = $imageInfo['mime'];
        $allowedMimes = ['image/jpeg', 'image/png', 'image/webp', 'image/gif'];
        
        if (!in_array($mimeType, $allowedMimes)) {
            Utils::respondJSON(['error' => 'Invalid image format'], 400);
        }
        
        // Generate unique filename
        $originalName = pathinfo($file['name'], PATHINFO_FILENAME);
        $safeName = preg_replace('/[^a-zA-Z0-9_-]/', '_', $originalName);
        $uniqueId = uniqid();
        $timestamp = date('YmdHis');
        $filename = "{$timestamp}_{$uniqueId}_{$safeName}.{$extension}";
        
        // Create upload directory if it doesn't exist
        $uploadPath = UPLOAD_DIR;
        if (!is_dir($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }
        
        $filePath = $uploadPath . $filename;
        
        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $filePath)) {
            Utils::respondJSON(['error' => 'Failed to save file'], 500);
        }
        
        // Create different sizes
        $sizes = [
            'thumbnail' => ['width' => THUMBNAIL_WIDTH, 'height' => THUMBNAIL_HEIGHT],
            'medium' => ['width' => LARGE_IMAGE_WIDTH, 'height' => LARGE_IMAGE_HEIGHT],
            'original' => null
        ];
        
        $imageUrls = [];
        
        foreach ($sizes as $sizeName => $dimensions) {
            if ($sizeName === 'original') {
                $imageUrls[$sizeName] = UPLOAD_URL . $filename;
            } else {
                $resizedFilename = "{$timestamp}_{$uniqueId}_{$safeName}_{$sizeName}.{$extension}";
                $resizedPath = $uploadPath . $resizedFilename;
                
                if (Utils::resizeImage($filePath, $resizedPath, $dimensions['width'], $dimensions['height'], IMAGE_QUALITY)) {
                    $imageUrls[$sizeName] = UPLOAD_URL . $resizedFilename;
                } else {
                    // If resize fails, use original
                    $imageUrls[$sizeName] = UPLOAD_URL . $filename;
                }
            }
        }
        
        // Log upload
        Utils::logError('Image uploaded successfully', [
            'filename' => $filename,
            'size' => $file['size'],
            'type' => $mimeType,
            'user_id' => JWT::validateToken($_SERVER['HTTP_AUTHORIZATION'])['user_id']
        ]);
        
        Utils::respondJSON([
            'success' => true,
            'filename' => $filename,
            'urls' => $imageUrls,
            'size' => $file['size'],
            'type' => $mimeType,
            'dimensions' => [
                'width' => $imageInfo[0],
                'height' => $imageInfo[1]
            ]
        ]);
        
    } catch (Exception $e) {
        Utils::logError('Upload error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Upload failed'], 500);
    }
}

function uploadMultipleImages() {
    try {
        if (!isset($_FILES['files']) || !is_array($_FILES['files']['name'])) {
            Utils::respondJSON(['error' => 'No files uploaded'], 400);
        }
        
        $files = $_FILES['files'];
        $uploadedFiles = [];
        $errors = [];
        
        $fileCount = count($files['name']);
        
        // Limit number of files
        if ($fileCount > 10) {
            Utils::respondJSON(['error' => 'Maximum 10 files allowed at once'], 400);
        }
        
        for ($i = 0; $i < $fileCount; $i++) {
            // Skip empty files
            if ($files['error'][$i] === UPLOAD_ERR_NO_FILE) {
                continue;
            }
            
            try {
                // Create a temporary $_FILES structure for single file processing
                $_FILES['file'] = [
                    'name' => $files['name'][$i],
                    'type' => $files['type'][$i],
                    'tmp_name' => $files['tmp_name'][$i],
                    'error' => $files['error'][$i],
                    'size' => $files['size'][$i]
                ];
                
                // Process single file (capture output)
                ob_start();
                uploadImage();
                $output = ob_get_clean();
                
                $result = json_decode($output, true);
                
                if ($result && isset($result['success']) && $result['success']) {
                    $uploadedFiles[] = $result;
                } else {
                    $errors[] = [
                        'filename' => $files['name'][$i],
                        'error' => $result['error'] ?? 'Upload failed'
                    ];
                }
                
            } catch (Exception $e) {
                $errors[] = [
                    'filename' => $files['name'][$i],
                    'error' => $e->getMessage()
                ];
            }
        }
        
        Utils::respondJSON([
            'success' => true,
            'uploaded' => $uploadedFiles,
            'errors' => $errors,
            'total_uploaded' => count($uploadedFiles),
            'total_errors' => count($errors)
        ]);
        
    } catch (Exception $e) {
        Utils::logError('Multiple upload error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Multiple upload failed'], 500);
    }
}

function deleteImage($imagePath) {
    try {
        // Sanitize the path
        $imagePath = basename($imagePath);
        
        // Check if file exists
        $fullPath = UPLOAD_DIR . $imagePath;
        
        if (!file_exists($fullPath)) {
            Utils::respondJSON(['error' => 'File not found'], 404);
        }
        
        // Security check - ensure file is in upload directory
        $realPath = realpath($fullPath);
        $uploadDirReal = realpath(UPLOAD_DIR);
        
        if (!$realPath || strpos($realPath, $uploadDirReal) !== 0) {
            Utils::respondJSON(['error' => 'Invalid file path'], 400);
        }
        
        // Delete the file
        if (unlink($fullPath)) {
            // Try to delete related sized versions
            $pathInfo = pathinfo($imagePath);
            $baseFilename = $pathInfo['filename'];
            $extension = $pathInfo['extension'];
            
            $sizes = ['thumbnail', 'medium'];
            foreach ($sizes as $size) {
                $sizedFile = UPLOAD_DIR . str_replace($pathInfo['filename'], $pathInfo['filename'] . '_' . $size, $imagePath);
                if (file_exists($sizedFile)) {
                    @unlink($sizedFile);
                }
            }
            
            Utils::respondJSON([
                'success' => true,
                'message' => 'File deleted successfully'
            ]);
        } else {
            Utils::respondJSON(['error' => 'Failed to delete file'], 500);
        }
        
    } catch (Exception $e) {
        Utils::logError('Delete image error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Delete failed'], 500);
    }
}

function getUploadedImages() {
    try {
        $images = [];
        $uploadDir = UPLOAD_DIR;
        
        if (!is_dir($uploadDir)) {
            Utils::respondJSON(['images' => []]);
        }
        
        $files = scandir($uploadDir);
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') continue;
            
            $filePath = $uploadDir . $file;
            
            if (is_file($filePath)) {
                $pathInfo = pathinfo($file);
                $extension = strtolower($pathInfo['extension'] ?? '');
                
                if (in_array($extension, ALLOWED_EXTENSIONS)) {
                    $fileSize = filesize($filePath);
                    $fileTime = filemtime($filePath);
                    
                    // Skip thumbnail and medium versions in listing
                    if (strpos($file, '_thumbnail.') !== false || strpos($file, '_medium.') !== false) {
                        continue;
                    }
                    
                    $images[] = [
                        'filename' => $file,
                        'url' => UPLOAD_URL . $file,
                        'size' => $fileSize,
                        'uploaded_at' => date('Y-m-d H:i:s', $fileTime),
                        'type' => $extension
                    ];
                }
            }
        }
        
        // Sort by upload time (newest first)
        usort($images, function($a, $b) {
            return strtotime($b['uploaded_at']) - strtotime($a['uploaded_at']);
        });
        
        Utils::respondJSON(['images' => $images]);
        
    } catch (Exception $e) {
        Utils::logError('Get images error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to get images'], 500);
    }
}

// Handle GET requests for listing images
if ($_SERVER['REQUEST_METHOD'] === 'GET' && !isset($id)) {
    // Validate admin token
    $token = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    $user = JWT::validateToken($token);
    
    if (!$user) {
        Utils::respondJSON(['error' => 'Unauthorized'], 401);
    }
    
    getUploadedImages();
}
?>