<?php


if (!defined('APP_ACCESS')) {
    die('Direct access not permitted');
}

function handleSettingsRequest($method, $action, $id, $data) {
    switch ($method) {
        case 'GET':
            getSettings();
            break;
            
        case 'PUT':
            updateSettings($data);
            break;
            
        default:
            Utils::respondJSON(['error' => 'Method not allowed'], 405);
    }
}

function getSettings() {
    try {
        $db = Database::getInstance()->getConnection();
        
        $sql = "SELECT setting_key, setting_value, description FROM app_settings ORDER BY setting_key";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $rows = $stmt->fetchAll();
        
        $settings = [];
        foreach ($rows as $row) {
            $settings[$row['setting_key']] = [
                'value' => $row['setting_value'],
                'description' => $row['description']
            ];
        }
        
        Utils::respondJSON(['settings' => $settings]);
        
    } catch (Exception $e) {
        Utils::logError('Get settings error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to get settings'], 500);
    }
}

function updateSettings($data) {
    // Validate admin token
    $token = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    $user = JWT::validateToken($token);
    
    if (!$user) {
        Utils::respondJSON(['error' => 'Unauthorized'], 401);
    }
    
    try {
        $db = Database::getInstance()->getConnection();
        
        $updatedSettings = [];
        
        foreach ($data as $key => $value) {
            // Sanitize the value
            $sanitizedValue = Utils::sanitizeInput($value);
            
            // Update or insert setting
            $sql = "INSERT INTO app_settings (setting_key, setting_value, updated_at) 
                    VALUES (?, ?, CURRENT_TIMESTAMP)
                    ON DUPLICATE KEY UPDATE 
                    setting_value = VALUES(setting_value), 
                    updated_at = VALUES(updated_at)";
            
            $stmt = $db->prepare($sql);
            $result = $stmt->execute([$key, $sanitizedValue]);
            
            if ($result) {
                $updatedSettings[$key] = $sanitizedValue;
            }
        }
        
        Utils::respondJSON([
            'success' => true,
            'updated_settings' => $updatedSettings,
            'message' => 'Settings updated successfully'
        ]);
        
    } catch (Exception $e) {
        Utils::logError('Update settings error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to update settings'], 500);
    }
}
?>