<?php

if (!defined('APP_ACCESS')) {
    die('Direct access not permitted');
}

// Include admin functions for auth
require_once 'admin.php';

function handleProductsRequest($method, $action, $id, $data) {
    $db = Database::getInstance()->getConnection();
    
    // Log the request details for debugging
    error_log("Products Request - Method: $method, Action: $action, ID: $id");
    
    switch ($method) {
        case 'GET':
            // GET requests don't require auth (public product viewing)
            if (!empty($id) && is_numeric($id)) {
                error_log("Getting single product with ID: $id");
                getProduct($db, $id);
            } else {
                error_log("Getting products list");
                getProducts($db, $data);
            }
            break;
            
        case 'POST':
            // POST requires admin auth (creating products)
            requireAdminAuth();
            createProduct($db, $data);
            break;
            
        case 'PUT':
            // PUT requires admin auth (updating products)
            requireAdminAuth();
            updateProduct($db, $id, $data);
            break;
            
        case 'DELETE':
            // DELETE requires admin auth (deleting products)
            requireAdminAuth();
            deleteProduct($db, $id);
            break;
            
        default:
            Utils::respondJSON(['error' => 'Method not allowed'], 405);
    }
}

function getProducts($db, $params) {
    try {
        // Track search if query provided
        if (!empty($params['search'])) {
            trackSearch($db, $params['search']);
        }
        
        $page = max(1, (int)($params['page'] ?? 1));
        $limit = min(MAX_PAGE_SIZE, max(1, (int)($params['limit'] ?? DEFAULT_PAGE_SIZE)));
        $offset = ($page - 1) * $limit;
        
        $where = ['p.is_active = 1'];
        $bindings = [];
        
        // Search functionality
        if (!empty($params['search'])) {
            $where[] = "(p.name LIKE ? OR p.description LIKE ? OR p.short_description LIKE ?)";
            $searchTerm = '%' . $params['search'] . '%';
            $bindings[] = $searchTerm;
            $bindings[] = $searchTerm;
            $bindings[] = $searchTerm;
        }
        
        // Category filter
        if (!empty($params['category_id'])) {
            $where[] = "p.category_id = ?";
            $bindings[] = $params['category_id'];
        }
        
        // Featured filter
        if (isset($params['featured']) && $params['featured'] !== '') {
            $where[] = "p.is_featured = ?";
            $bindings[] = $params['featured'] ? 1 : 0;
        }
        
        // Price range
        if (!empty($params['min_price'])) {
            $where[] = "p.price >= ?";
            $bindings[] = $params['min_price'];
        }
        
        if (!empty($params['max_price'])) {
            $where[] = "p.price <= ?";
            $bindings[] = $params['max_price'];
        }
        
        $whereClause = implode(' AND ', $where);
        
        // Get total count
        $countSql = "SELECT COUNT(*) as total 
                     FROM products p 
                     LEFT JOIN categories c ON p.category_id = c.id 
                     WHERE $whereClause";
        
        $countStmt = $db->prepare($countSql);
        $countStmt->execute($bindings);
        $total = $countStmt->fetch()['total'];
        
        // Get products
        $orderBy = "ORDER BY ";
        switch ($params['sort'] ?? 'default') {
            case 'price_asc':
                $orderBy .= "p.price ASC";
                break;
            case 'price_desc':
                $orderBy .= "p.price DESC";
                break;
            case 'name_asc':
                $orderBy .= "p.name ASC";
                break;
            case 'name_desc':
                $orderBy .= "p.name DESC";
                break;
            case 'newest':
                $orderBy .= "p.created_at DESC";
                break;
            default:
                $orderBy .= "p.is_featured DESC, p.sort_order ASC, p.created_at DESC";
        }
        
        $sql = "SELECT p.*, c.name as category_name,
                       COUNT(pv.id) as view_count
                FROM products p 
                LEFT JOIN categories c ON p.category_id = c.id
                LEFT JOIN product_views pv ON p.id = pv.product_id AND pv.viewed_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                WHERE $whereClause
                GROUP BY p.id
                $orderBy
                LIMIT ? OFFSET ?";
        
        $stmt = $db->prepare($sql);
        $stmt->execute([...$bindings, $limit, $offset]);
        $products = $stmt->fetchAll();
        
        // Process products data
        foreach ($products as &$product) {
            $product['images'] = json_decode($product['images'], true) ?: [];
            $product['price'] = Utils::formatPrice($product['price']);
            $product['compare_price'] = $product['compare_price'] ? Utils::formatPrice($product['compare_price']) : null;
            
            // Get variations
            $product['variations'] = getProductVariations($db, $product['id']);
            
            // Calculate discount percentage
            if ($product['compare_price'] && floatval($product['compare_price']) > floatval($product['price'])) {
                $discount = (floatval($product['compare_price']) - floatval($product['price'])) / floatval($product['compare_price']) * 100;
                $product['discount_percentage'] = round($discount);
            }
        }
        
        Utils::respondJSON([
            'products' => $products,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => (int)$total,
                'pages' => ceil($total / $limit),
                'has_next' => $page < ceil($total / $limit),
                'has_prev' => $page > 1
            ]
        ]);
        
    } catch (Exception $e) {
        Utils::logError('Error getting products: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to fetch products'], 500);
    }
}

function getProduct($db, $id) {
    try {
        error_log("Fetching product with ID: $id");
        
        // Validate ID
        if (!is_numeric($id) || $id <= 0) {
            Utils::respondJSON(['error' => 'Invalid product ID'], 400);
            return;
        }
        
        $sql = "SELECT p.*, c.name as category_name,
                       COUNT(pv.id) as view_count
                FROM products p 
                LEFT JOIN categories c ON p.category_id = c.id
                LEFT JOIN product_views pv ON p.id = pv.product_id AND pv.viewed_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                WHERE p.id = ? AND p.is_active = 1
                GROUP BY p.id";
        
        $stmt = $db->prepare($sql);
        $stmt->execute([$id]);
        $product = $stmt->fetch();
        
        if (!$product) {
            error_log("Product not found with ID: $id");
            Utils::respondJSON(['error' => 'Product not found'], 404);
            return;
        }
        
        error_log("Found product: " . $product['name']);
        
        // Process product data
        $product['images'] = json_decode($product['images'], true) ?: [];
        $product['price'] = Utils::formatPrice($product['price']);
        $product['compare_price'] = $product['compare_price'] ? Utils::formatPrice($product['compare_price']) : null;
        
        // Get variations
        $product['variations'] = getProductVariations($db, $product['id']);
        
        // Get related products (same category, exclude current)
        $relatedSql = "SELECT id, name, price, images, short_description 
                       FROM products 
                       WHERE category_id = ? AND id != ? AND is_active = 1 
                       ORDER BY is_featured DESC, RAND() 
                       LIMIT 4";
        
        $relatedStmt = $db->prepare($relatedSql);
        $relatedStmt->execute([$product['category_id'], $id]);
        $relatedProducts = $relatedStmt->fetchAll();
        
        foreach ($relatedProducts as &$related) {
            $related['images'] = json_decode($related['images'], true) ?: [];
            $related['price'] = Utils::formatPrice($related['price']);
        }
        
        $product['related_products'] = $relatedProducts;
        
        // Calculate discount percentage
        if ($product['compare_price'] && floatval($product['compare_price']) > floatval($product['price'])) {
            $discount = (floatval($product['compare_price']) - floatval($product['price'])) / floatval($product['compare_price']) * 100;
            $product['discount_percentage'] = round($discount);
        }
        
        error_log("Returning product data for ID: $id");
        Utils::respondJSON(['product' => $product]);
        
    } catch (Exception $e) {
        error_log('Error getting product: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to fetch product'], 500);
    }
}

function getProductVariations($db, $productId) {
    $sql = "SELECT * FROM product_variations 
            WHERE product_id = ? AND is_active = 1 
            ORDER BY name, sort_order ASC";
    
    $stmt = $db->prepare($sql);
    $stmt->execute([$productId]);
    $variations = $stmt->fetchAll();
    
    // Group variations by name
    $groupedVariations = [];
    foreach ($variations as $variation) {
        $name = $variation['name'];
        if (!isset($groupedVariations[$name])) {
            $groupedVariations[$name] = [];
        }
        $groupedVariations[$name][] = [
            'id' => $variation['id'],
            'value' => $variation['value'],
            'price_modifier' => Utils::formatPrice($variation['price_modifier']),
            'stock_quantity' => $variation['stock_quantity'],
            'sku' => $variation['sku']
        ];
    }
    
    return $groupedVariations;
}

function createProduct($db, $data) {
    try {
        $required = ['name', 'price'];
        $missing = Utils::validateRequiredFields($data, $required);
        
        if (!empty($missing)) {
            Utils::respondJSON(['error' => 'Missing required fields: ' . implode(', ', $missing)], 400);
        }
        
        // Validate price
        if (!is_numeric($data['price']) || floatval($data['price']) <= 0) {
            Utils::respondJSON(['error' => 'Valid price is required'], 400);
        }
        
        // Generate SKU if not provided
        $sku = $data['sku'] ?? strtoupper(Utils::createSlug($data['name']) . '-' . substr(Utils::generateUUID(), 0, 8));
        
        // Ensure SKU is unique
        $skuCheckSql = "SELECT COUNT(*) as count FROM products WHERE sku = ?";
        $skuCheckStmt = $db->prepare($skuCheckSql);
        $skuCheckStmt->execute([$sku]);
        if ($skuCheckStmt->fetch()['count'] > 0) {
            $sku = $sku . '-' . time();
        }
        
        $sql = "INSERT INTO products (
                    name, description, short_description, price, compare_price, sku, 
                    category_id, images, is_active, is_featured, stock_quantity, 
                    manage_stock, weight, dimensions, meta_title, meta_description, sort_order
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $db->prepare($sql);
        $result = $stmt->execute([
            Utils::sanitizeInput($data['name']),
            Utils::sanitizeInput($data['description'] ?? ''),
            Utils::sanitizeInput($data['short_description'] ?? ''),
            floatval($data['price']),
            isset($data['compare_price']) && $data['compare_price'] ? floatval($data['compare_price']) : null,
            $sku,
            !empty($data['category_id']) ? intval($data['category_id']) : null,
            json_encode($data['images'] ?? []),
            isset($data['is_active']) ? ($data['is_active'] ? 1 : 0) : 1,
            isset($data['is_featured']) ? ($data['is_featured'] ? 1 : 0) : 0,
            intval($data['stock_quantity'] ?? 0),
            isset($data['manage_stock']) ? ($data['manage_stock'] ? 1 : 0) : 1,
            isset($data['weight']) && $data['weight'] ? floatval($data['weight']) : null,
            Utils::sanitizeInput($data['dimensions'] ?? ''),
            Utils::sanitizeInput($data['meta_title'] ?? ''),
            Utils::sanitizeInput($data['meta_description'] ?? ''),
            intval($data['sort_order'] ?? 0)
        ]);
        
        if ($result) {
            $productId = $db->lastInsertId();
            
            // Add variations if provided
            if (!empty($data['variations']) && is_array($data['variations'])) {
                Utils::logError('Adding variations for product', [
                    'product_id' => $productId,
                    'variations_count' => count($data['variations'])
                ]);
                addProductVariations($db, $productId, $data['variations']);
            }
            
            Utils::logError('Product created successfully', [
                'product_id' => $productId,
                'name' => $data['name']
            ]);
            
            Utils::respondJSON([
                'success' => true,
                'product_id' => intval($productId),
                'message' => 'Product created successfully'
            ], 201);
        } else {
            Utils::respondJSON(['error' => 'Failed to create product'], 500);
        }
        
    } catch (Exception $e) {
        Utils::logError('Error creating product: ' . $e->getMessage(), [
            'data' => $data
        ]);
        Utils::respondJSON(['error' => 'Failed to create product: ' . $e->getMessage()], 500);
    }
}

function updateProduct($db, $id, $data) {
    try {
        // Validate ID
        if (!is_numeric($id) || intval($id) <= 0) {
            Utils::respondJSON(['error' => 'Invalid product ID'], 400);
        }
        
        // Check if product exists
        $checkSql = "SELECT id FROM products WHERE id = ?";
        $checkStmt = $db->prepare($checkSql);
        $checkStmt->execute([$id]);
        
        if (!$checkStmt->fetch()) {
            Utils::respondJSON(['error' => 'Product not found'], 404);
        }
        
        $updateFields = [];
        $values = [];
        
        $allowedFields = [
            'name', 'description', 'short_description', 'price', 'compare_price', 
            'sku', 'category_id', 'images', 'is_active', 'is_featured', 
            'stock_quantity', 'manage_stock', 'weight', 'dimensions', 
            'meta_title', 'meta_description', 'sort_order'
        ];
        
        foreach ($allowedFields as $field) {
            if (array_key_exists($field, $data)) {
                $updateFields[] = "$field = ?";
                $value = $data[$field];
                
                // Special handling for different field types
                switch ($field) {
                    case 'name':
                    case 'description':
                    case 'short_description':
                    case 'meta_title':
                    case 'meta_description':
                    case 'dimensions':
                        $value = Utils::sanitizeInput($value);
                        break;
                        
                    case 'price':
                        if (!is_numeric($value) || floatval($value) <= 0) {
                            Utils::respondJSON(['error' => 'Valid price is required'], 400);
                        }
                        $value = floatval($value);
                        break;
                        
                    case 'compare_price':
                        $value = ($value && is_numeric($value)) ? floatval($value) : null;
                        break;
                        
                    case 'category_id':
                        $value = $value ? intval($value) : null;
                        break;
                        
                    case 'stock_quantity':
                    case 'sort_order':
                        $value = intval($value);
                        break;
                        
                    case 'weight':
                        $value = ($value && is_numeric($value)) ? floatval($value) : null;
                        break;
                        
                    case 'images':
                        $value = json_encode(is_array($value) ? $value : []);
                        break;
                        
                    case 'is_active':
                    case 'is_featured':
                    case 'manage_stock':
                        $value = $value ? 1 : 0;
                        break;
                }
                
                $values[] = $value;
            }
        }
        
        if (empty($updateFields)) {
            Utils::respondJSON(['error' => 'No valid fields to update'], 400);
        }
        
        $values[] = intval($id);
        
        $sql = "UPDATE products SET " . implode(', ', $updateFields) . ", updated_at = CURRENT_TIMESTAMP WHERE id = ?";
        $stmt = $db->prepare($sql);
        $result = $stmt->execute($values);
        
        if ($result) {
            // Handle variations update if provided
            if (array_key_exists('variations', $data)) {
                // Delete existing variations
                $deleteSql = "DELETE FROM product_variations WHERE product_id = ?";
                $deleteStmt = $db->prepare($deleteSql);
                $deleteStmt->execute([$id]);
                
                // Add new variations
                if (!empty($data['variations']) && is_array($data['variations'])) {
                    Utils::logError('Updating variations for product', [
                        'product_id' => $id,
                        'variations_count' => count($data['variations'])
                    ]);
                    addProductVariations($db, $id, $data['variations']);
                }
            }
            
            Utils::logError('Product updated successfully', [
                'product_id' => $id,
                'updated_fields' => array_keys($data)
            ]);
            
            Utils::respondJSON([
                'success' => true,
                'message' => 'Product updated successfully'
            ]);
        } else {
            Utils::respondJSON(['error' => 'Failed to update product'], 500);
        }
        
    } catch (Exception $e) {
        Utils::logError('Error updating product: ' . $e->getMessage(), [
            'product_id' => $id,
            'data' => $data
        ]);
        Utils::respondJSON(['error' => 'Failed to update product: ' . $e->getMessage()], 500);
    }
}

function deleteProduct($db, $id) {
    try {
        // Soft delete - just mark as inactive
        $sql = "UPDATE products SET is_active = 0, updated_at = CURRENT_TIMESTAMP WHERE id = ?";
        $stmt = $db->prepare($sql);
        $result = $stmt->execute([$id]);
        
        if ($result && $stmt->rowCount() > 0) {
            Utils::respondJSON([
                'success' => true,
                'message' => 'Product deleted successfully'
            ]);
        } else {
            Utils::respondJSON(['error' => 'Product not found'], 404);
        }
        
    } catch (Exception $e) {
        Utils::logError('Error deleting product: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to delete product'], 500);
    }
}

function addProductVariations($db, $productId, $variations) {
    if (empty($variations) || !is_array($variations)) {
        return;
    }
    
    $sql = "INSERT INTO product_variations (product_id, name, value, price_modifier, stock_quantity, sku, sort_order, is_active) 
            VALUES (?, ?, ?, ?, ?, ?, ?, 1)";
    $stmt = $db->prepare($sql);
    
    foreach ($variations as $index => $variation) {
        try {
            if (empty($variation['name']) || empty($variation['value'])) {
                Utils::logError('Skipping invalid variation', $variation);
                continue;
            }
            
            $stmt->execute([
                intval($productId),
                Utils::sanitizeInput($variation['name']),
                Utils::sanitizeInput($variation['value']),
                floatval($variation['price_modifier'] ?? 0),
                intval($variation['stock_quantity'] ?? 0),
                Utils::sanitizeInput($variation['sku'] ?? ''),
                intval($variation['sort_order'] ?? $index)
            ]);
            
        } catch (Exception $e) {
            Utils::logError('Error adding variation: ' . $e->getMessage(), [
                'product_id' => $productId,
                'variation' => $variation
            ]);
            // Continue with other variations even if one fails
        }
    }
}

function trackSearch($db, $query) {
    try {
        // Count results for this search
        $countSql = "SELECT COUNT(*) as total FROM products p 
                     WHERE p.is_active = 1 
                     AND (p.name LIKE ? OR p.description LIKE ? OR p.short_description LIKE ?)";
        $searchTerm = '%' . $query . '%';
        
        $countStmt = $db->prepare($countSql);
        $countStmt->execute([$searchTerm, $searchTerm, $searchTerm]);
        $results = $countStmt->fetch()['total'];
        
        // Insert search query with result count
        $sql = "INSERT INTO search_queries (query, results_count, ip_address) VALUES (?, ?, ?)";
        $stmt = $db->prepare($sql);
        $stmt->execute([
            Utils::sanitizeInput($query),
            $results,
            Utils::getClientIP()
        ]);
    } catch (Exception $e) {
        // Don't fail the main request if tracking fails
        Utils::logError('Error tracking search: ' . $e->getMessage());
    }
}
?>