<?php

if (!defined('APP_ACCESS')) {
    die('Direct access not permitted');
}

function handleCategoriesRequest($method, $action, $id, $data) {
    $db = Database::getInstance()->getConnection();
    
    switch ($method) {
        case 'GET':
            if ($id) {
                getCategory($db, $id);
            } else {
                getCategories($db, $data);
            }
            break;
            
        case 'POST':
            createCategory($db, $data);
            break;
            
        case 'PUT':
            updateCategory($db, $id, $data);
            break;
            
        case 'DELETE':
            deleteCategory($db, $id);
            break;
            
        default:
            Utils::respondJSON(['error' => 'Method not allowed'], 405);
    }
}

function getCategories($db, $params) {
    try {
        $sql = "SELECT c.*, 
                       COUNT(p.id) as product_count
                FROM categories c
                LEFT JOIN products p ON c.id = p.category_id AND p.is_active = 1
                WHERE c.is_active = 1
                GROUP BY c.id
                ORDER BY c.sort_order ASC, c.name ASC";
        
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $categories = $stmt->fetchAll();
        
        Utils::respondJSON(['categories' => $categories]);
        
    } catch (Exception $e) {
        Utils::logError('Error getting categories: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to fetch categories'], 500);
    }
}

function getCategory($db, $id) {
    try {
        $sql = "SELECT c.*, 
                       COUNT(p.id) as product_count
                FROM categories c
                LEFT JOIN products p ON c.id = p.category_id AND p.is_active = 1
                WHERE c.id = ? AND c.is_active = 1
                GROUP BY c.id";
        
        $stmt = $db->prepare($sql);
        $stmt->execute([$id]);
        $category = $stmt->fetch();
        
        if (!$category) {
            Utils::respondJSON(['error' => 'Category not found'], 404);
        }
        
        Utils::respondJSON(['category' => $category]);
        
    } catch (Exception $e) {
        Utils::logError('Error getting category: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to fetch category'], 500);
    }
}

function createCategory($db, $data) {
    // Validate admin token
    $token = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    $user = JWT::validateToken($token);
    
    if (!$user) {
        Utils::respondJSON(['error' => 'Unauthorized'], 401);
    }
    
    try {
        $required = ['name'];
        $missing = Utils::validateRequiredFields($data, $required);
        
        if (!empty($missing)) {
            Utils::respondJSON(['error' => 'Missing required fields: ' . implode(', ', $missing)], 400);
        }
        
        $sql = "INSERT INTO categories (name, description, image_url, is_active, sort_order) 
                VALUES (?, ?, ?, ?, ?)";
        
        $stmt = $db->prepare($sql);
        $result = $stmt->execute([
            Utils::sanitizeInput($data['name']),
            Utils::sanitizeInput($data['description'] ?? ''),
            $data['image_url'] ?? null,
            $data['is_active'] ?? true,
            $data['sort_order'] ?? 0
        ]);
        
        if ($result) {
            $categoryId = $db->lastInsertId();
            Utils::respondJSON([
                'success' => true,
                'category_id' => $categoryId,
                'message' => 'Category created successfully'
            ], 201);
        } else {
            Utils::respondJSON(['error' => 'Failed to create category'], 500);
        }
        
    } catch (Exception $e) {
        Utils::logError('Error creating category: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to create category'], 500);
    }
}

function updateCategory($db, $id, $data) {
    // Validate admin token
    $token = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    $user = JWT::validateToken($token);
    
    if (!$user) {
        Utils::respondJSON(['error' => 'Unauthorized'], 401);
    }
    
    try {
        // Check if category exists
        $checkSql = "SELECT id FROM categories WHERE id = ?";
        $checkStmt = $db->prepare($checkSql);
        $checkStmt->execute([$id]);
        
        if (!$checkStmt->fetch()) {
            Utils::respondJSON(['error' => 'Category not found'], 404);
        }
        
        $updateFields = [];
        $values = [];
        
        $allowedFields = ['name', 'description', 'image_url', 'is_active', 'sort_order'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updateFields[] = "$field = ?";
                $value = $data[$field];
                
                if (in_array($field, ['name', 'description'])) {
                    $value = Utils::sanitizeInput($value);
                }
                
                $values[] = $value;
            }
        }
        
        if (empty($updateFields)) {
            Utils::respondJSON(['error' => 'No valid fields to update'], 400);
        }
        
        $values[] = $id;
        
        $sql = "UPDATE categories SET " . implode(', ', $updateFields) . ", updated_at = CURRENT_TIMESTAMP WHERE id = ?";
        $stmt = $db->prepare($sql);
        $result = $stmt->execute($values);
        
        if ($result) {
            Utils::respondJSON([
                'success' => true,
                'message' => 'Category updated successfully'
            ]);
        } else {
            Utils::respondJSON(['error' => 'Failed to update category'], 500);
        }
        
    } catch (Exception $e) {
        Utils::logError('Error updating category: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to update category'], 500);
    }
}

function deleteCategory($db, $id) {
    // Validate admin token
    $token = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    $user = JWT::validateToken($token);
    
    if (!$user) {
        Utils::respondJSON(['error' => 'Unauthorized'], 401);
    }
    
    try {
        // Check if category has products
        $checkSql = "SELECT COUNT(*) as count FROM products WHERE category_id = ? AND is_active = 1";
        $checkStmt = $db->prepare($checkSql);
        $checkStmt->execute([$id]);
        $count = $checkStmt->fetch()['count'];
        
        if ($count > 0) {
            Utils::respondJSON(['error' => 'Cannot delete category with active products'], 400);
        }
        
        // Soft delete - mark as inactive
        $sql = "UPDATE categories SET is_active = 0, updated_at = CURRENT_TIMESTAMP WHERE id = ?";
        $stmt = $db->prepare($sql);
        $result = $stmt->execute([$id]);
        
        if ($result && $stmt->rowCount() > 0) {
            Utils::respondJSON([
                'success' => true,
                'message' => 'Category deleted successfully'
            ]);
        } else {
            Utils::respondJSON(['error' => 'Category not found'], 404);
        }
        
    } catch (Exception $e) {
        Utils::logError('Error deleting category: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to delete category'], 500);
    }
}
?>