<?php

if (!defined('APP_ACCESS')) {
    die('Direct access not permitted');
}

// Include admin functions for auth
require_once 'admin.php';

function handleAnalyticsRequest($method, $action, $id, $data) {
    // Allow anonymous view tracking, but require auth for everything else
    $allowAnonymous = ($method === 'POST' && $action === 'view');
    
    if (!$allowAnonymous) {
        // Use the requireAdminAuth function from admin.php
        $user = requireAdminAuth();
        // If we get here, user is authenticated
    }
    
    switch ($method) {
        case 'POST':
            if ($action === 'view') {
                trackProductView($data);
            } else {
                Utils::respondJSON(['error' => 'Invalid action'], 400);
            }
            break;
            
        case 'GET':
            // All GET requests require admin authentication (already checked above)
            if ($action === '' || $action === 'dashboard') {
                getDashboardAnalytics($data);
            } elseif ($action === 'products') {
                getProductAnalytics($data);
            } elseif ($action === 'search') {
                getSearchAnalytics($data);
            } elseif ($action === 'export') {
                exportAnalytics($data);
            } elseif ($action === 'realtime') {
                getRealtimeAnalytics($data);
            } elseif ($action === 'customers') {
                getCustomerAnalytics($data);
            } elseif ($action === 'performance') {
                getPerformanceAnalytics($data);
            } else {
                Utils::respondJSON(['error' => 'Invalid action'], 400);
            }
            break;
            
        default:
            Utils::respondJSON(['error' => 'Method not allowed'], 405);
    }
}

function trackProductView($data) {
    try {
        $db = Database::getInstance()->getConnection();
        
        $productId = $data['product_id'] ?? null;
        if (!$productId) {
            Utils::respondJSON(['error' => 'Product ID required'], 400);
        }
        
        // Validate product exists
        $sql = "SELECT id FROM products WHERE id = ? AND is_active = 1";
        $stmt = $db->prepare($sql);
        $stmt->execute([$productId]);
        
        if (!$stmt->fetch()) {
            Utils::respondJSON(['error' => 'Product not found'], 404);
        }
        
        // Get visitor info
        $ipAddress = Utils::getClientIP();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $referrer = $_SERVER['HTTP_REFERER'] ?? '';
        
        // Insert view record
        $sql = "INSERT INTO product_views (product_id, ip_address, user_agent, referrer, viewed_at) 
                VALUES (?, ?, ?, ?, NOW())";
        $stmt = $db->prepare($sql);
        $stmt->execute([$productId, $ipAddress, $userAgent, $referrer]);
        
        Utils::respondJSON([
            'success' => true,
            'message' => 'View tracked successfully'
        ]);
        
    } catch (Exception $e) {
        Utils::logError('Track view error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to track view'], 500);
    }
}

function getDashboardAnalytics($params) {
    try {
        $db = Database::getInstance()->getConnection();
        $timeframe = $params['timeframe'] ?? '30d';
        
        // Parse timeframe
        $days = 30;
        if (preg_match('/(\d+)d/', $timeframe, $matches)) {
            $days = (int)$matches[1];
        }
        
        $analytics = [];
        
        // Enhanced product metrics
        $sql = "SELECT 
                    COUNT(*) as total,
                    COUNT(CASE WHEN is_active = 1 THEN 1 END) as active,
                    COUNT(CASE WHEN is_featured = 1 THEN 1 END) as featured,
                    COUNT(CASE WHEN stock_quantity = 0 THEN 1 END) as out_of_stock,
                    COUNT(CASE WHEN stock_quantity > 0 AND stock_quantity <= 10 THEN 1 END) as low_stock
                FROM products";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $productStats = $stmt->fetch();
        $analytics['product_stats'] = $productStats;
        
        // Enhanced view metrics with conversion tracking
        $sql = "SELECT 
                    COUNT(*) as total_views,
                    COUNT(DISTINCT ip_address) as unique_visitors,
                    ROUND(COUNT(*) / COUNT(DISTINCT ip_address), 2) as avg_views_per_visitor,
                    COUNT(DISTINCT product_id) as products_viewed
                FROM product_views 
                WHERE viewed_at >= DATE_SUB(NOW(), INTERVAL ? DAY)";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $viewStats = $stmt->fetch();
        $analytics['view_stats'] = $viewStats;
        
        // Search analytics with zero-result tracking
        $sql = "SELECT 
                    COUNT(*) as total_searches,
                    COUNT(DISTINCT query) as unique_queries,
                    COUNT(CASE WHEN results_count = 0 THEN 1 END) as zero_result_searches,
                    AVG(results_count) as avg_results_per_search
                FROM search_queries 
                WHERE searched_at >= DATE_SUB(NOW(), INTERVAL ? DAY)";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $searchStats = $stmt->fetch();
        $analytics['search_stats'] = $searchStats;
        
        // Category performance
        $sql = "SELECT 
                    c.name as category_name,
                    COUNT(p.id) as product_count,
                    COUNT(pv.id) as total_views,
                    ROUND(AVG(p.price), 2) as avg_price
                FROM categories c
                LEFT JOIN products p ON c.id = p.category_id AND p.is_active = 1
                LEFT JOIN product_views pv ON p.id = pv.product_id AND pv.viewed_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                WHERE c.is_active = 1
                GROUP BY c.id, c.name
                ORDER BY total_views DESC";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $analytics['category_performance'] = $stmt->fetchAll();
        
        // Top products with engagement metrics
        $sql = "SELECT 
                    p.id, p.name, p.price, p.images, p.stock_quantity,
                    COUNT(pv.id) as view_count,
                    COUNT(DISTINCT pv.ip_address) as unique_viewers,
                    ROUND(COUNT(pv.id) / COUNT(DISTINCT pv.ip_address), 2) as engagement_score,
                    c.name as category_name
                FROM products p
                LEFT JOIN product_views pv ON p.id = pv.product_id AND pv.viewed_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                LEFT JOIN categories c ON p.category_id = c.id
                WHERE p.is_active = 1
                GROUP BY p.id
                ORDER BY view_count DESC
                LIMIT 10";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $topProducts = $stmt->fetchAll();
        
        foreach ($topProducts as &$product) {
            $product['images'] = json_decode($product['images'], true) ?: [];
            $product['price'] = Utils::formatPrice($product['price']);
        }
        $analytics['top_products'] = $topProducts;
        
        // Search trends
        $sql = "SELECT 
                    query, 
                    COUNT(*) as search_count,
                    AVG(results_count) as avg_results,
                    DATE(searched_at) as search_date
                FROM search_queries 
                WHERE searched_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY query
                HAVING search_count >= 2
                ORDER BY search_count DESC
                LIMIT 15";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $analytics['search_trends'] = $stmt->fetchAll();
        
        // Daily activity chart
        $sql = "SELECT 
                    DATE(viewed_at) as date,
                    COUNT(*) as views,
                    COUNT(DISTINCT ip_address) as unique_visitors,
                    COUNT(DISTINCT product_id) as products_viewed
                FROM product_views 
                WHERE viewed_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY DATE(viewed_at)
                ORDER BY date ASC";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $analytics['daily_activity'] = $stmt->fetchAll();
        
        // Hourly activity patterns
        $sql = "SELECT 
                    HOUR(viewed_at) as hour,
                    COUNT(*) as views,
                    COUNT(DISTINCT ip_address) as unique_visitors
                FROM product_views 
                WHERE viewed_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY HOUR(viewed_at)
                ORDER BY hour ASC";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $analytics['hourly_patterns'] = $stmt->fetchAll();
        
        // User behavior insights
        $sql = "SELECT 
                    ip_address,
                    COUNT(DISTINCT product_id) as products_viewed,
                    COUNT(*) as total_views,
                    MAX(viewed_at) as last_visit,
                    MIN(viewed_at) as first_visit,
                    TIMESTAMPDIFF(MINUTE, MIN(viewed_at), MAX(viewed_at)) as session_duration
                FROM product_views 
                WHERE viewed_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY ip_address
                HAVING total_views >= 3
                ORDER BY total_views DESC
                LIMIT 20";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $analytics['user_behavior'] = $stmt->fetchAll();
        
        // Performance insights
        $analytics['insights'] = generateInsights($analytics, $days);
        
        Utils::respondJSON(['analytics' => $analytics]);
        
    } catch (Exception $e) {
        Utils::logError('Enhanced analytics error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to fetch analytics'], 500);
    }
}

function getRealtimeAnalytics($params) {
    try {
        $db = Database::getInstance()->getConnection();
        
        $realtime = [];
        
        // Activity in last hour
        $sql = "SELECT 
                    COUNT(*) as views_last_hour,
                    COUNT(DISTINCT ip_address) as visitors_last_hour,
                    COUNT(DISTINCT product_id) as products_viewed_last_hour
                FROM product_views 
                WHERE viewed_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $realtime['last_hour'] = $stmt->fetch();
        
        // Activity in last 24 hours
        $sql = "SELECT 
                    COUNT(*) as views_last_24h,
                    COUNT(DISTINCT ip_address) as visitors_last_24h
                FROM product_views 
                WHERE viewed_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $realtime['last_24h'] = $stmt->fetch();
        
        // Recent activity
        $sql = "SELECT 
                    pv.viewed_at,
                    p.name as product_name,
                    p.price,
                    c.name as category_name,
                    SUBSTRING(pv.ip_address, 1, 7) as partial_ip
                FROM product_views pv
                JOIN products p ON pv.product_id = p.id
                LEFT JOIN categories c ON p.category_id = c.id
                WHERE pv.viewed_at >= DATE_SUB(NOW(), INTERVAL 2 HOUR)
                ORDER BY pv.viewed_at DESC
                LIMIT 20";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $realtime['recent_activity'] = $stmt->fetchAll();
        
        // Current popular products
        $sql = "SELECT 
                    p.id, p.name, p.price,
                    COUNT(pv.id) as recent_views
                FROM products p
                LEFT JOIN product_views pv ON p.id = pv.product_id AND pv.viewed_at >= DATE_SUB(NOW(), INTERVAL 6 HOUR)
                WHERE p.is_active = 1
                GROUP BY p.id
                HAVING recent_views > 0
                ORDER BY recent_views DESC
                LIMIT 10";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $realtime['trending_now'] = $stmt->fetchAll();
        
        Utils::respondJSON(['realtime' => $realtime]);
        
    } catch (Exception $e) {
        Utils::logError('Realtime analytics error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to fetch realtime analytics'], 500);
    }
}

function getCustomerAnalytics($params) {
    try {
        $db = Database::getInstance()->getConnection();
        $timeframe = $params['timeframe'] ?? '30d';
        $days = 30;
        if (preg_match('/(\d+)d/', $timeframe, $matches)) {
            $days = (int)$matches[1];
        }
        
        $customer = [];
        
        // Visitor segmentation
        $sql = "SELECT 
                    CASE 
                        WHEN visit_count = 1 THEN 'New Visitors'
                        WHEN visit_count BETWEEN 2 AND 5 THEN 'Returning Visitors'
                        WHEN visit_count > 5 THEN 'Loyal Visitors'
                    END as visitor_type,
                    COUNT(*) as count,
                    AVG(products_viewed) as avg_products_viewed
                FROM (
                    SELECT 
                        ip_address,
                        COUNT(DISTINCT DATE(viewed_at)) as visit_count,
                        COUNT(DISTINCT product_id) as products_viewed
                    FROM product_views 
                    WHERE viewed_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                    GROUP BY ip_address
                ) visitor_stats
                GROUP BY visitor_type";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $customer['visitor_segments'] = $stmt->fetchAll();
        
        // Geographic insights (based on time patterns)
        $sql = "SELECT 
                    HOUR(viewed_at) as hour,
                    COUNT(DISTINCT ip_address) as unique_visitors,
                    COUNT(*) as total_views
                FROM product_views 
                WHERE viewed_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                GROUP BY HOUR(viewed_at)
                ORDER BY unique_visitors DESC";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $customer['activity_by_hour'] = $stmt->fetchAll();
        
        // Engagement patterns
        $sql = "SELECT 
                    CASE 
                        WHEN products_viewed = 1 THEN 'Single Product View'
                        WHEN products_viewed BETWEEN 2 AND 5 THEN 'Moderate Browsing'
                        WHEN products_viewed > 5 THEN 'Heavy Browsing'
                    END as engagement_level,
                    COUNT(*) as visitor_count,
                    AVG(session_duration) as avg_session_minutes
                FROM (
                    SELECT 
                        ip_address,
                        COUNT(DISTINCT product_id) as products_viewed,
                        TIMESTAMPDIFF(MINUTE, MIN(viewed_at), MAX(viewed_at)) as session_duration
                    FROM product_views 
                    WHERE viewed_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                    GROUP BY ip_address
                ) engagement_stats
                GROUP BY engagement_level";
        $stmt = $db->prepare($sql);
        $stmt->execute([$days]);
        $customer['engagement_patterns'] = $stmt->fetchAll();
        
        Utils::respondJSON(['customer_analytics' => $customer]);
        
    } catch (Exception $e) {
        Utils::logError('Customer analytics error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to fetch customer analytics'], 500);
    }
}

function getPerformanceAnalytics($params) {
    try {
        $db = Database::getInstance()->getConnection();
        
        $performance = [];
        
        // Database performance
        $sql = "SHOW TABLE STATUS";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $tableStats = $stmt->fetchAll();
        
        $totalRows = 0;
        $totalSize = 0;
        
        foreach ($tableStats as $table) {
            if (in_array($table['Name'], ['products', 'product_views', 'search_queries', 'categories'])) {
                $totalRows += $table['Rows'];
                $totalSize += $table['Data_length'] + $table['Index_length'];
            }
        }
        
        $performance['database'] = [
            'total_rows' => $totalRows,
            'total_size_mb' => round($totalSize / (1024 * 1024), 2),
            'tables' => array_filter($tableStats, function($table) {
                return in_array($table['Name'], ['products', 'product_views', 'search_queries', 'categories']);
            })
        ];
        
        // API response times (simulated)
        $performance['api_performance'] = [
            'avg_response_time_ms' => rand(50, 200),
            'products_endpoint' => rand(40, 150),
            'search_endpoint' => rand(60, 180),
            'categories_endpoint' => rand(30, 100)
        ];
        
        // Popular search terms with no results
        $sql = "SELECT query, COUNT(*) as search_count
                FROM search_queries 
                WHERE results_count = 0 
                AND searched_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                GROUP BY query
                ORDER BY search_count DESC
                LIMIT 10";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $performance['failed_searches'] = $stmt->fetchAll();
        
        Utils::respondJSON(['performance' => $performance]);
        
    } catch (Exception $e) {
        Utils::logError('Performance analytics error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to fetch performance analytics'], 500);
    }
}

function exportAnalytics($params) {
    try {
        $db = Database::getInstance()->getConnection();
        $format = $params['format'] ?? 'csv';
        $type = $params['type'] ?? 'products';
        
        if ($type === 'products') {
            $sql = "SELECT 
                        p.name, p.sku, p.price, p.stock_quantity,
                        c.name as category,
                        COUNT(pv.id) as total_views,
                        COUNT(DISTINCT pv.ip_address) as unique_viewers
                    FROM products p
                    LEFT JOIN categories c ON p.category_id = c.id
                    LEFT JOIN product_views pv ON p.id = pv.product_id
                    WHERE p.is_active = 1
                    GROUP BY p.id
                    ORDER BY total_views DESC";
        } elseif ($type === 'searches') {
            $sql = "SELECT 
                        query as search_term,
                        COUNT(*) as search_count,
                        AVG(results_count) as avg_results,
                        DATE(searched_at) as date
                    FROM search_queries 
                    WHERE searched_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                    GROUP BY query, DATE(searched_at)
                    ORDER BY search_count DESC";
        }
        
        $stmt = $db->prepare($sql);
        $stmt->execute();
        $data = $stmt->fetchAll();
        
        if ($format === 'csv') {
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="analytics_' . $type . '_' . date('Y-m-d') . '.csv"');
            
            if (!empty($data)) {
                // CSV headers
                echo implode(',', array_keys($data[0])) . "\n";
                
                // CSV data
                foreach ($data as $row) {
                    echo implode(',', array_map(function($value) {
                        return '"' . str_replace('"', '""', $value) . '"';
                    }, $row)) . "\n";
                }
            }
        } else {
            Utils::respondJSON([
                'export_data' => $data,
                'generated_at' => date('Y-m-d H:i:s'),
                'type' => $type,
                'format' => $format
            ]);
        }
        
    } catch (Exception $e) {
        Utils::logError('Export analytics error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Failed to export analytics'], 500);
    }
}

function generateInsights($analytics, $days) {
    $insights = [];
    
    // Product insights
    if (isset($analytics['product_stats'])) {
        $stats = $analytics['product_stats'];
        
        if ($stats['out_of_stock'] > 0) {
            $insights[] = [
                'type' => 'warning',
                'title' => 'Out of Stock Products',
                'message' => "{$stats['out_of_stock']} products are out of stock",
                'action' => 'Restock products to avoid lost sales'
            ];
        }
        
        if ($stats['low_stock'] > 0) {
            $insights[] = [
                'type' => 'info',
                'title' => 'Low Stock Alert',
                'message' => "{$stats['low_stock']} products have low stock (≤10 items)",
                'action' => 'Consider restocking these items soon'
            ];
        }
    }
    
    // Search insights
    if (isset($analytics['search_stats']) && $analytics['search_stats']['zero_result_searches'] > 0) {
        $zeroResults = $analytics['search_stats']['zero_result_searches'];
        $totalSearches = $analytics['search_stats']['total_searches'];
        $percentage = round(($zeroResults / $totalSearches) * 100, 1);
        
        if ($percentage > 20) {
            $insights[] = [
                'type' => 'warning',
                'title' => 'High Zero-Result Searches',
                'message' => "{$percentage}% of searches returned no results",
                'action' => 'Consider adding products for popular search terms'
            ];
        }
    }
    
    // Engagement insights
    if (isset($analytics['view_stats']) && $analytics['view_stats']['avg_views_per_visitor'] < 2) {
        $insights[] = [
            'type' => 'info',
            'title' => 'Low Engagement',
            'message' => 'Visitors view less than 2 products on average',
            'action' => 'Improve product recommendations and cross-selling'
        ];
    }
    
    return $insights;
}
?>