<?php

if (!defined('APP_ACCESS')) {
    die('Direct access not permitted');
}

function handleAdminRequest($method, $action, $id, $data) {
    switch ($method) {
        case 'POST':
            if ($action === 'login') {
                adminLogin($data);
            } else {
                Utils::respondJSON(['error' => 'Invalid action'], 400);
            }
            break;
            
        case 'GET':
            if ($action === 'verify-token' || $action === 'verify') {
                verifyAdminToken();
            } else {
                Utils::respondJSON(['error' => 'Invalid action'], 400);
            }
            break;
            
        default:
            Utils::respondJSON(['error' => 'Method not allowed'], 405);
    }
}

function adminLogin($data) {
    try {
        $required = ['username', 'password'];
        $missing = Utils::validateRequiredFields($data, $required);
        
        if (!empty($missing)) {
            Utils::respondJSON(['error' => 'Missing required fields: ' . implode(', ', $missing)], 400);
        }
        
        $db = Database::getInstance()->getConnection();
        
        $sql = "SELECT id, username, email, password_hash, full_name, role, is_active 
                FROM admin_users 
                WHERE username = ? AND is_active = 1";
        
        $stmt = $db->prepare($sql);
        $stmt->execute([Utils::sanitizeInput($data['username'])]);
        $user = $stmt->fetch();
        
        if (!$user || !password_verify($data['password'], $user['password_hash'])) {
            // Log failed login attempt
            Utils::logError('Failed login attempt', [
                'username' => $data['username'],
                'ip' => Utils::getClientIP()
            ]);
            
            Utils::respondJSON(['error' => 'Invalid credentials'], 401);
        }
        
        // Update last login
        $updateSql = "UPDATE admin_users SET last_login = CURRENT_TIMESTAMP WHERE id = ?";
        $updateStmt = $db->prepare($updateSql);
        $updateStmt->execute([$user['id']]);
        
        // Generate JWT token
        $token = JWT::createToken($user['id'], $user['username']);
        
        // Log successful login
        Utils::logError('Successful admin login', [
            'user_id' => $user['id'],
            'username' => $user['username'],
            'ip' => Utils::getClientIP()
        ]);
        
        Utils::respondJSON([
            'success' => true,
            'token' => $token,
            'user' => [
                'id' => $user['id'],
                'username' => $user['username'],
                'email' => $user['email'],
                'full_name' => $user['full_name'],
                'role' => $user['role']
            ]
        ]);
        
    } catch (Exception $e) {
        Utils::logError('Admin login error: ' . $e->getMessage());
        Utils::respondJSON(['error' => 'Login failed'], 500);
    }
}

function getAuthorizationHeader() {
    $headers = null;
    
    // Try different ways to get Authorization header
    if (isset($_SERVER['Authorization'])) {
        $headers = trim($_SERVER['Authorization']);
    } else if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
        $headers = trim($_SERVER['HTTP_AUTHORIZATION']);
    } else if (function_exists('apache_request_headers')) {
        $requestHeaders = apache_request_headers();
        // Server-side fix for bug in older PHP versions where 'Authorization' is not set
        $requestHeaders = array_combine(array_map('ucwords', array_map('strtolower', array_keys($requestHeaders))), array_values($requestHeaders));
        if (isset($requestHeaders['Authorization'])) {
            $headers = trim($requestHeaders['Authorization']);
        }
    }
    
    // Check for token in different places
    if (empty($headers)) {
        // Check if token is in query parameter (fallback)
        if (isset($_GET['token'])) {
            $headers = 'Bearer ' . $_GET['token'];
        }
        // Check if token is in POST data (fallback)
        else if (isset($_POST['token'])) {
            $headers = 'Bearer ' . $_POST['token'];
        }
        // Check for custom header
        else if (isset($_SERVER['HTTP_X_AUTHORIZATION'])) {
            $headers = trim($_SERVER['HTTP_X_AUTHORIZATION']);
        }
    }
    
    return $headers;
}

function verifyAdminToken() {
    try {
        // Get token using the improved method
        $authHeader = getAuthorizationHeader();
        
        // Debug log
        Utils::logError('Token verification attempt', [
            'auth_header_exists' => !empty($authHeader),
            'server_vars' => [
                'HTTP_AUTHORIZATION' => $_SERVER['HTTP_AUTHORIZATION'] ?? 'not set',
                'Authorization' => $_SERVER['Authorization'] ?? 'not set',
            ],
            'ip' => Utils::getClientIP()
        ]);
        
        if (empty($authHeader)) {
            Utils::respondJSON(['error' => 'No authorization header provided', 'valid' => false], 401);
        }
        
        // Extract token (remove "Bearer " prefix if present)
        $token = $authHeader;
        if (strpos($token, 'Bearer ') === 0) {
            $token = substr($token, 7);
        }
        
        // Validate token using JWT class
        $tokenData = JWT::validateToken($token);
        
        if (!$tokenData) {
            Utils::logError('Token validation failed', [
                'token_length' => strlen($token),
                'ip' => Utils::getClientIP()
            ]);
            Utils::respondJSON(['error' => 'Invalid or expired token', 'valid' => false], 401);
        }
        
        // Verify user still exists and is active
        $db = Database::getInstance()->getConnection();
        
        $sql = "SELECT id, username, email, full_name, role, is_active, last_login
                FROM admin_users 
                WHERE id = ? AND is_active = 1";
        
        $stmt = $db->prepare($sql);
        $stmt->execute([$tokenData['user_id']]);
        $userData = $stmt->fetch();
        
        if (!$userData) {
            Utils::logError('Token verification failed - user not found or inactive', [
                'user_id' => $tokenData['user_id'],
                'ip' => Utils::getClientIP()
            ]);
            Utils::respondJSON(['error' => 'User not found or inactive', 'valid' => false], 404);
        }
        
        // Log successful verification
        Utils::logError('Token verification successful', [
            'user_id' => $userData['id'],
            'username' => $userData['username'],
            'ip' => Utils::getClientIP()
        ]);
        
        Utils::respondJSON([
            'valid' => true,
            'user' => [
                'id' => $userData['id'],
                'username' => $userData['username'],
                'email' => $userData['email'],
                'full_name' => $userData['full_name'],
                'role' => $userData['role'],
                'last_login' => $userData['last_login']
            ]
        ]);
        
    } catch (Exception $e) {
        Utils::logError('Token verification error: ' . $e->getMessage(), [
            'ip' => Utils::getClientIP(),
            'auth_header_provided' => !empty($authHeader)
        ]);
        Utils::respondJSON(['error' => 'Token verification failed', 'valid' => false], 500);
    }
}

function requireAdminAuth() {
    $authHeader = getAuthorizationHeader();
    
    if (empty($authHeader)) {
        Utils::respondJSON(['error' => 'Authentication required'], 401);
    }
    
    $token = $authHeader;
    if (strpos($token, 'Bearer ') === 0) {
        $token = substr($token, 7);
    }
    
    $tokenData = JWT::validateToken($token);
    
    if (!$tokenData) {
        Utils::respondJSON(['error' => 'Invalid or expired token'], 401);
    }
    
    // Verify user is still active
    $db = Database::getInstance()->getConnection();
    $sql = "SELECT id, username, role FROM admin_users WHERE id = ? AND is_active = 1";
    $stmt = $db->prepare($sql);
    $stmt->execute([$tokenData['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        Utils::respondJSON(['error' => 'User not found or inactive'], 401);
    }
    
    return $user;
}

// Helper function to get current admin user from token
function getCurrentAdminUser() {
    try {
        $authHeader = getAuthorizationHeader();
        
        if (empty($authHeader)) {
            return null;
        }
        
        $token = $authHeader;
        if (strpos($token, 'Bearer ') === 0) {
            $token = substr($token, 7);
        }
        
        $tokenData = JWT::validateToken($token);
        
        if (!$tokenData) {
            return null;
        }
        
        $db = Database::getInstance()->getConnection();
        $sql = "SELECT id, username, email, full_name, role FROM admin_users WHERE id = ? AND is_active = 1";
        $stmt = $db->prepare($sql);
        $stmt->execute([$tokenData['user_id']]);
        
        return $stmt->fetch();
        
    } catch (Exception $e) {
        Utils::logError('Get current admin user error: ' . $e->getMessage());
        return null;
    }
}

// Initialize admin tables if they don't exist
function initializeAdminTables() {
    try {
        $db = Database::getInstance()->getConnection();
        
        // Check if admin_users table exists
        $sql = "SHOW TABLES LIKE 'admin_users'";
        $stmt = $db->prepare($sql);
        $stmt->execute();
        
        if ($stmt->rowCount() === 0) {
            // Create admin_users table
            $createTableSQL = "
                CREATE TABLE admin_users (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    username VARCHAR(50) NOT NULL UNIQUE,
                    email VARCHAR(100) NOT NULL UNIQUE,
                    password_hash VARCHAR(255) NOT NULL,
                    full_name VARCHAR(100),
                    role ENUM('admin', 'manager', 'editor') DEFAULT 'admin',
                    is_active BOOLEAN DEFAULT TRUE,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    last_login TIMESTAMP NULL,
                    INDEX idx_username (username),
                    INDEX idx_email (email),
                    INDEX idx_active (is_active)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
            ";
            
            $db->exec($createTableSQL);
            
            // Create default admin user
            $defaultPassword = 'admin123'; // Change this in production!
            $hashedPassword = password_hash($defaultPassword, PASSWORD_DEFAULT);
            
            $insertAdminSQL = "
                INSERT INTO admin_users (username, email, password_hash, full_name, role) 
                VALUES ('admin', 'admin@preorderstore.com', ?, 'Store Administrator', 'admin')
            ";
            
            $stmt = $db->prepare($insertAdminSQL);
            $stmt->execute([$hashedPassword]);
            
            Utils::logError('Admin tables initialized with default admin user', [
                'username' => 'admin',
                'password' => $defaultPassword
            ]);
        }
        
    } catch (Exception $e) {
        Utils::logError('Failed to initialize admin tables: ' . $e->getMessage());
    }
}

// Call initialization on first load
if (!isset($GLOBALS['admin_tables_initialized'])) {
    initializeAdminTables();
    $GLOBALS['admin_tables_initialized'] = true;
}

?>